<?php

/**
 * @copyright softwarewerk <info@softwarewerk.de>
 * @author    chris <chris@softwarewerk.de>
 * Creationtime: 00:07 - 04.04.21
 */

namespace cu\DataImport\Validator;

use cu\DataImport\ITranslator;
use cu\DataImport\Translator;

/**
 * Class AbstractValidator
 *
 * @package cu\DataImport\Validator
 */
abstract class AbstractValidator implements IValidator
{
    /**
     * @var array<string,mixed>
     */
    protected $options;

    /**
     * @var ITranslator
     */
    protected $translator;

    /**
     * @var ?string
     */
    protected $message;

    /**
     * AbstractValidator constructor.
     *
     * @param ?array<string,mixed> $options
     */
    public function __construct(?array $options = null)
    {
        $this->setOptions($options);
    }

    /**
     * @return ITranslator
     */
    public function initTranslator(): ITranslator
    {
        if (!isset($this->translator)) {
            $this->translator = Translator::initTranslator();
        }

        return $this->translator;
    }

    /**
     * @param ITranslator $translator
     * @return AbstractValidator
     */
    public function setTranslator(ITranslator $translator): AbstractValidator
    {
        $this->translator = $translator;

        return $this;
    }

    /**
     * @return AbstractValidator
     */
    protected function reset(): AbstractValidator
    {
        $this->message = null;

        return $this;
    }

    /**
     * @param string $messageKey
     * @param ?array<mixed> $data
     */
    protected function raiseError(string $messageKey, ?array $data = null): void
    {
        $this->message = $this->initTranslator()->translateKey(
            $messageKey,
            $data
        );
    }

    /**
     * @inheritDoc
     */
    public function getMessage(): ?string
    {
        return $this->message;
    }

    /**
     * @param string $name
     * @param mixed $default
     *
     * @return mixed
     */
    public function get(string $name, $default = null)
    {
        return $this->options[$name] ?: $default;
    }

    /**
     * @param ?array<string,mixed> $options
     *
     * @return AbstractValidator
     */
    public function setOptions(?array $options = null): AbstractValidator
    {
        if (empty($options)) {
            $this->options = [];
        } else {
            $this->options = array_filter(
                $options,
                function ($val, $key) {
                    $method = "set" . ucfirst($key);
                    if (method_exists($this, $method)) {
                        if (is_array($val)) {
                            $this->{$method}(...$val);
                        } else {
                            $this->{$method}($val);
                        }


                        return true;
                    }

                    return false;
                },
                ARRAY_FILTER_USE_BOTH
            );
        }

        return $this;
    }
}
