<?php

/**
 * @copyright softwarewerk <info@softwarewerk.de>
 * @author    chris <chris@softwarewerk.de>
 * Creationtime: 22:54 - 03.04.21
 */

namespace cu\DataImport;

use cu\DataImport\DataInput\IDataInput;
use cu\DataImport\DataSource\IDataSource;
use cu\DataImport\Validator\IValidator;

/**
 * Class Import
 *
 * @package cu\DataImport
 */
class Import
{
    public const MODE_IMPORT_VALID_ROWS = 'IMPORT_VALID_ROWS';
    public const MODE_ONE_INVALID_ROW_CANCELS_IMPORT = 'ONE_INVALID_ROW_CANCELS_IMPORT';

    /**
     * @var string
     */
    private $importMode = self::MODE_ONE_INVALID_ROW_CANCELS_IMPORT;

    /**
     * @var IDataSource
     */
    private $dataSource;

    /**
     * @var IDataInput
     */
    private $dataInput;

    /**
     * @var ITranslator
     */
    private $translator;

    /**
     * @var array<string,array<IValidator>>
     */
    private $validators = [];

    /**
     * @var int
     */
    private $rowIndex = 0;

    /**
     * @var bool
     */
    private $allRowsValid = true;

    /**
     * @var array<?string>
     */
    private $errors = [];

    /**
     * @return ITranslator
     */
    public function initTranslator(): ITranslator
    {
        if (!isset($this->translator)) {
            $this->translator = Translator::initTranslator();
        }

        return $this->translator;
    }

    /**
     * @param ITranslator $translator
     * @return Import
     */
    public function setTranslator(ITranslator $translator): Import
    {
        $this->translator = $translator;

        return $this;
    }

    /**
     * @param array<string,array> $row
     * @return bool
     */
    protected function checkData(/* Traversable */ array $row): bool
    {
        if (empty($row)) {
            $this->errors[] = sprintf('No valid data found in row %s. Skipping.', $this->rowIndex);

            return false;
        }

        $invalid = false;
        foreach ($row as $columnIdentifier => $colInfo) {
            if (empty($this->validators[$columnIdentifier])) {
                continue;
            }

            foreach ($this->validators[$columnIdentifier] as $validator) {
                if (!$validator->validate($colInfo['value'], $this->rowIndex, $colInfo['colIndex'])) {
                    $invalid = true;
                    $this->errors[] = $validator->getMessage();
                }
            }
        }

        if (!$invalid) {
            return true;
        }

        $this->allRowsValid = false;

        return false;
    }

    /**
     * @param string $importMode
     * @return Import
     */
    public function setImportMode(string $importMode): Import
    {
        $this->importMode = $importMode;

        return $this;
    }

    /**
     * @return string
     */
    public function getImportMode(): string
    {
        return $this->importMode;
    }

    /**
     * @param IDataSource $dataSource
     *
     * @return Import
     */
    public function setDataSource(IDataSource $dataSource): Import
    {
        $this->dataSource = $dataSource;

        return $this;
    }

    /**
     * @param IDataInput $dataInput
     * @return Import
     */
    public function setDataInput(IDataInput $dataInput): Import
    {
        $this->dataInput = $dataInput;

        return $this;
    }

    /**
     * @param string $forColumnIdentifier
     * @param IValidator $validator
     * @return Import
     */
    public function registerValidatorInstance(string $forColumnIdentifier, IValidator $validator): Import
    {
        if (!isset($this->validators[$forColumnIdentifier])) {
            $this->validators[$forColumnIdentifier] = [];
        }

        $this->validators[$forColumnIdentifier][] = $validator;

        return $this;
    }

    /**
     * @return array<?string>
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * @return Import
     * @throws DataImportException
     */
    public function import(): Import
    {
        $this->allRowsValid = true;
        $this->rowIndex = 0;
        $this->errors = [];

        if (!isset($this->dataSource)) {
            throw new DataImportException("Datasource missing.");
        }

        if (!isset($this->dataInput)) {
            throw new DataImportException("Data input missing.");
        }

        $this->dataInput->begin();

        foreach ($this->dataSource->getIterable() as $rowIndex => $row) {
            $this->rowIndex++;

            $rowIsValid = $this->checkData($row);
            if (!$rowIsValid) {
                continue;
            }

            /* @phpstan-ignore-next-line */
            if (!$this->allRowsValid && $this->getImportMode() === static::MODE_ONE_INVALID_ROW_CANCELS_IMPORT) {
                // if we found one invalid row and the import mode is MODE_ONE_INVALID_ROW_CANCELS_IMPORT
                // we omit all subsequent imports as we will cancel the import anyway.
                continue;
            }

            $data = array_map(
                function (array $cellInfo) {
                    return $cellInfo['value'];
                },
                $row
            );

            try {
                $this->dataInput->import($data);
            } catch (\Exception $e) {
                // we clear all error messages first as they play no role in this situation
                $this->errors = [];

                // TODO: translate
                $this->errors[] = 'Import failed.';

                // TODO: introduce logging

                return $this;
            }
        }

        /* @phpstan-ignore-next-line */
        if ($this->allRowsValid || $this->getImportMode() === static::MODE_IMPORT_VALID_ROWS) {
            $this->dataInput->flush();

            return $this;
        }

        // TODO: Warning

        // Import failed at some point - e.g. because we found a row containing invalid data.
        // -> cancel
        /* @phpstan-ignore-next-line */
        $this->dataInput->cancel();

        return $this;
    }
}
