<?php

/**
 * @copyright softwarewerk <info@softwarewerk.de>
 * @author    chris <chris@softwarewerk.de>
 * Creationtime: 22:54 - 03.04.21
 */

namespace cu\DataImport\DataSource;

use cu\DataImport\DataImportException;
use cu\DataImport\DataSource\FileUploadAdapter\IFileUploadAdapter;

/**
 * Class UploadedFileDataSource
 *
 * @package cu\DataImport\DataSource
 */
class UploadedFileDataSource implements IDataSource
{
    /**
     * @var IFileUploadAdapter
     */
    private $adapter;

    /**
     * @var string
     */
    private $name;

    /**
     * @var string
     */
    private $uploadDirectory;

    /**
     * UploadedFileDataSource constructor.
     *
     * @param IFileUploadAdapter $adapter
     * @param string $name
     * @param string $uploadDirectory
     * @throws DataImportException
     */
    public function __construct(IFileUploadAdapter $adapter, string $name, string $uploadDirectory)
    {
        $this->name = $name;

        $uploadDirectory = realpath($uploadDirectory);
        if (!$uploadDirectory) {
            throw new DataImportException('Invalid path');
        }

        $this->uploadDirectory = rtrim($uploadDirectory, DIRECTORY_SEPARATOR);

        $this->setAdapter($adapter);
    }

    /**
     * @param IFileUploadAdapter $dataSource
     */
    public function setAdapter(IFileUploadAdapter $dataSource): void
    {
        $this->adapter = $dataSource;
    }

    /**
     * @throws DataImportException
     */
    public function receiveUploadedFile(): void
    {
        if (empty($_FILES[$this->name])) {
            return;
        }

        $uploadedFileInfo = $_FILES[$this->name];
        if (is_array($uploadedFileInfo['name'])) {
            throw new DataImportException('Upload of multiple files detected. This is not supported yet');
        }

        if (!empty($uploadedFileInfo['error']) && UPLOAD_ERR_OK !== $uploadedFileInfo['error']) {
            throw new DataImportException('Upload of multiple files detected. This is not supported yet');
        }

        if (!is_file($uploadedFileInfo['tmp_name']) || !is_readable($uploadedFileInfo['tmp_name'])) {
            throw new DataImportException('Invalid upload.');
        }

        $type = $uploadedFileInfo['type'];

        if (empty($type)) {
            throw new DataImportException('Unable to determine the type of the uploaded file.');
        }

        if (empty($this->adapter->canHandle($type))) {
            throw new DataImportException(
                sprintf(
                    "The provided adapter reported that it cannot handle the uploaded file type '%s'.",
                    $type
                )
            );
        }

        $filename = $this->adapter->assembleFilename($this->name);
        if (empty($filename)) {
            throw new DataImportException(
                sprintf(
                    "The provided adapter did not properly assemble the required file name of '%s'.",
                    $this->name
                )
            );
        }

        $file = $this->uploadDirectory . DIRECTORY_SEPARATOR . $filename;

        if (!is_dir($this->uploadDirectory)) {
            throw new DataImportException(sprintf('Invalid upload directory %s', $this->uploadDirectory));
        }

        if (!is_writable($this->uploadDirectory)) {
            throw new DataImportException(
                sprintf('Unable to move uploaded file to %s. Insufficient permission', $this->uploadDirectory)
            );
        }

        if (!move_uploaded_file($uploadedFileInfo['tmp_name'], $file)) {
            throw new DataImportException(
                sprintf('Unable to move uploaded file to %s.', $this->uploadDirectory)
            );
        }
    }

    /**
     * @inheritDoc
     */
    public function getIterable(): \Traversable
    {
        $this->receiveUploadedFile();

        $filename = $this->adapter->assembleFilename($this->name);
        if (empty($filename)) {
            throw new DataImportException(
                sprintf(
                    "The provided adapter did not properly assemble the required file name of '%s'.",
                    $this->name
                )
            );
        }

        $file = $this->uploadDirectory . DIRECTORY_SEPARATOR . $filename;
        if (!is_file($file) || !is_readable($file)) {
            throw new DataImportException(sprintf('Invalid source %s', $file));
        }

        $this->adapter->setFile($file);

        return $this->adapter->getIterable();
    }
}
