<?php

/**
 * @copyright softwarewerk <info@softwarewerk.de>
 * @author    chris <chris@softwarewerk.de>
 * Creationtime: 22:54 - 03.04.21
 */

namespace cu\DataImport\DataSource;

use cu\DataImport\DataImportException;

/**
 * Class CSVFileDataSource
 *
 * @package cu\DataImport\DataSource
 * @implements \Iterator<array>
 */
class CSVFileDataSource extends AbstractDataSource implements IFileDataSource, \Iterator
{
    public const CSV_DELIMITER_SEMICOLON = ";";
    public const CSV_DELIMITER_COLON = ";";
    public const CSV_ENCLOSURE_SINGLE_QUOTE = "'";
    public const CSV_ENCLOSURE_DOUBLE_QUOTE = "\"";

    /**
     * @var string
     */
    protected static $defaultDelimiter = self::CSV_DELIMITER_COLON;

    /**
     * @var ?string
     */
    protected static $defaultEnclosure = null;

    /**
     * @var string
     */
    private $sourceFile;

    /**
     * @var ?string
     */
    private $delimiter;

    /**
     * @var ?string
     */
    private $enclosure;

    /**
     * @var ?resource
     */
    private $resourceHandle;

    /**
     * @var int
     */
    private $lineNumber = 1;

    /**
     * @var array<array>
     */
    private $currentValue;

    /**
     * @var bool
     */
    private $skipFirstLine = false;

    /**
     * CSVFileDataSource constructor.
     *
     * @param ?string $sourceFile
     * @param ?string $delimiter
     * @param ?string $enclosure
     */
    public function __construct(
        ?string $sourceFile = null,
        ?string $delimiter = null,
        ?string $enclosure = null
    ) {
        if ($sourceFile) {
            $this->setFile($sourceFile);
        }

        $this->delimiter = $delimiter ?? static::$defaultDelimiter;
        $this->enclosure = $enclosure ?? static::$defaultEnclosure;
    }

    /**
     * @param string $sourceFile
     *
     * @return void
     */
    public function setFile(string $sourceFile): void
    {
        $this->sourceFile = $sourceFile;
    }

    /**
     * @param ?bool $flag
     * @return CSVFileDataSource
     */
    public function setSkipFirstLine(?bool $flag = true): CSVFileDataSource
    {
        $this->skipFirstLine = (true === $flag);

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getIterable(): \Traversable
    {
        return $this;
    }

    /**
     * @inheritDoc
     * @return array<string,array>
     */
    public function current(): array
    {
        return $this->currentValue;
    }

    /**
     * @inheritDoc
     */
    public function next(): void
    {
        $this->lineNumber++;
    }

    /**
     * @inheritDoc
     * @return int
     */
    public function key(): int
    {
        return $this->lineNumber;
    }

    /**
     * @inheritDoc
     */
    public function valid(): bool
    {
        $this->currentValue = [];

        if (!is_resource($this->resourceHandle)) {
            return false;
        }

        // If we got no col spec then iterating the data source makes no sense.
        if (empty($this->colSpec)) {
            return false;
        }

        if (!feof($this->resourceHandle)) {
            $line = fgetcsv($this->resourceHandle, 0, ';');
            if (null !== $line && false !== $line) {
                $values = array_values($line);
                foreach ($this->colSpec as $colName => $colSpec) {
                    $colIndex = @$colSpec['colIndex'];

                    // treat not existing columns as if they were empty...
                    // TODO: this could cause confusion - rethink!
                    //  1) Option 1 - ignore non existing columns
                    //  2) Option 2 - introduce logging and log incident
                    //  3) Option 3 - treat as severe error condition and throw an error
                    $value = isset($values[$colIndex]) ? $values[$colIndex] : null;

                    $this->currentValue[$colName] = $this->createColInfo(
                        $colIndex,
                        $this->runFormatter($colName, $value)
                    );
                }

                return true;
            }
        }

        @fclose($this->resourceHandle);

        unset($this->resourceHandle);

        return false;
    }

    /**
     * @inheritDoc
     */
    public function rewind(): void
    {
        if (empty($this->sourceFile) || !is_readable($this->sourceFile)) {
            throw new DataImportException('Source file not readable.');
        }

        if (!empty($this->resourceHandle)) {
            @fclose($this->resourceHandle);
            unset($this->resourceHandle);
        }

        $this->resourceHandle = fopen($this->sourceFile, 'r') ?: null;
        $this->lineNumber = 1;

        if (!is_resource($this->resourceHandle)) {
            throw new DataImportException('Failed to open file.');
        }

        if ($this->skipFirstLine) {
            fgets($this->resourceHandle);
            $this->lineNumber++;
        }
    }
}
