<?php
/**
 * @copyright softwarewerk <info@softwarewerk.de>
 * @author chris <chris@softwarewerk.de>
 * Creationtime: 22:41 - 09.04.21
 */

namespace cu\DataImport\DataSource;

use cu\DataImport\DataImportException;
use cu\DataImport\Formatter\IFormatter;

/**
 * Class AbstractDataSource
 *
 * @package cu\DataImport\DataSource
 */
abstract class AbstractDataSource implements IDataSource
{
    /**
     * @var array<string,array>
     */
    protected $colSpec;

    /**
     * @var array<string,(IFormatter|callable)>
     */
    private $formatters = [];

    /**
     * @param string $colIdentifier
     * @param IFormatter $formatter
     * @return AbstractDataSource
     */
    public function registerFormatterInstance(string $colIdentifier, IFormatter $formatter): AbstractDataSource
    {
        $this->formatters[$colIdentifier] = $formatter;

        return $this;
    }

    /**
     * @param string $colIdentifier
     * @param callable $formatter
     * @return AbstractDataSource
     */
    public function registerFormatterCallable(string $colIdentifier, callable $formatter): AbstractDataSource
    {
        $this->formatters[$colIdentifier] = $formatter;

        return $this;
    }

    /**
     * @param string $colIdentifier
     * @param mixed $value
     * @return mixed
     */
    protected function runFormatter(string $colIdentifier, $value)
    {
        $formatter = null;
        if (!empty($this->formatters[$colIdentifier])) {
            $formatter = $this->formatters[$colIdentifier];
        } elseif (!empty($this->colSpec[$colIdentifier]['formatter'])) {
            $formatterClassName = $this->colSpec[$colIdentifier]['formatter'];
            if (class_exists($formatterClassName)) {
                $formatter = $this->formatters[$colIdentifier] = new $formatterClassName();
            }
        }

        if ($formatter instanceof IFormatter) {
            return $formatter->format($value);
        } elseif (is_callable($formatter)) {
            return call_user_func($formatter, $value);
        }

        return $value;
    }

    /**
     * @param ?array<(int|string),(string|int|array<string,mixed>)> $colSpec
     * @throws DataImportException
     */
    public function selectColumns(?array $colSpec): void
    {
        $this->colSpec = $this->assembleColSpec($colSpec);
    }

    /**
     * @param ?array<(int|string),(string|int|array<string,mixed>)> $initialColSpec
     * @return array<string,array<string,mixed>>
     * @throws DataImportException
     */
    protected function assembleColSpec(?array $initialColSpec): array
    {
        $finalColSpec = [];
        $colIndex = 1;
        /* @phpstan-ignore-next-line */
        foreach ($initialColSpec as $colName => $colSpec) {
            if (is_string($colName)) {
                if (is_int($colSpec)) {
                    $finalColSpec[$colName] = [
                        'colIndex' => $colIndex = $colSpec
                    ];
                } elseif (is_array($colSpec)) {
                    if (!isset($colSpec['colIndex'])) {
                        $colSpec['colIndex'] = $colIndex;
                    } else {
                        $colSpec['colIndex'] = $colIndex = (int)$colSpec['colIndex'];
                    }

                    $finalColSpec[$colName] = $colSpec;
                } else {
                    throw new DataImportException('Invalid col spec.');
                }
            } elseif (is_string($colSpec)) {
                $finalColSpec[$colSpec] = [
                    'colIndex' => $colIndex
                ];
            } else {
                throw new DataImportException('Invalid col spec.');
            }

            $colIndex += 1;
        }

        return $finalColSpec;
    }

    /**
     * @param int $colIndex
     * @param mixed $value
     * @return array<string,int|mixed>
     */
    protected function createColInfo(int $colIndex, $value): array
    {
        return [
            'colIndex' => $colIndex,
            'value' => $value
        ];
    }
}
