<?php
/**
 * @copyright softwarewerk <info@softwarewerk.de>
 * @author    chris <chris@softwarewerk.de>
 * Creationtime: 22:54 - 03.04.21
 */

namespace cu\DataImport\DataSource;

use cu\DataImport\DataImportException;

/**
 * Class CSVFileDataSource
 *
 * @package cu\DataImport\DataSource
 */
class CSVFileDataSource implements IFileDataSource, \Iterator
{
    /**
     * @var string
     */
    private $sourceFile;

    /**
     * @var resource
     */
    private $resourceHandle;

    /**
     * @var int
     */
    private $lineNumber = 0;

    /**
     * @var array<string>
     */
    private $currentValue;


    /**
     * CSVFileDataSource constructor.
     *
     * @param ?string $sourceFile
     */
    public function __construct(?string $sourceFile)
    {
        if ($sourceFile) {
            $this->setFile($sourceFile);
        }
    }

    /**
     * @param $sourceFile
     *
     * @return void
     */
    public function setFile(string $sourceFile): void
    {
        $this->sourceFile = $sourceFile;
    }

    /**
     * @inheritDoc
     */
    public function getIterable(): \Traversable
    {
        return $this;
    }

    /**
     * @param string $fileType
     * @return bool
     */
    public function canHandle(string $fileType): bool
    {
        return 'csv' === $fileType;
    }

    /**
     * @inheritDoc
     */
    public function current(): array
    {
        return $this->currentValue;
    }

    /**
     * @inheritDoc
     */
    public function next()
    {
    }

    /**
     * @inheritDoc
     */
    public function key()
    {
        return $this->lineNumber;
    }

    /**
     * @inheritDoc
     */
    public function valid(): bool
    {
        if (!is_resource($this->resourceHandle)) {
            return false;
        }

        if (!feof($this->resourceHandle)) {
            $line = fgetcsv($this->resourceHandle);
            if (null !== $line && false !== $line) {
                $this->currentValue = $line;
                $this->lineNumber++;
                return true;
            }
        }

        @fclose($this->resourceHandle);
        return false;
    }

    /**
     * @inheritDoc
     */
    public function rewind()
    {
        if (empty($this->sourceFile) || !is_readable($this->sourceFile)) {
            throw new DataImportException('Source file not readable.');
        }

        if (!empty($this->resourceHandle)) {
            @fclose($this->resourceHandle);
            unset($this->resourceHandle);
        }

        $this->resourceHandle = fopen($this->sourceFile, 'r');
        $this->lineNumber = 0;
    }
}
