<?php
/**
 * @copyright softwarewerk <info@softwarewerk.de>
 * @author    chris <chris@softwarewerk.de>
 * Creationtime: 22:54 - 03.04.21
 */

namespace cu\DataImport;

use cu\DataImport\DataInput\IDataInput;
use cu\DataImport\DataSource\IDataSource;
use cu\DataImport\Validator\IValidator;

/**
 * Class Import
 *
 * @package CuCommon
 */
class Import
{
    public const MODE_IMPORT_VALID_ROWS = 'IMPORT_VALID_ROWS';
    public const MODE_ONE_INVALID_ROW_CANCELS_IMPORT = 'ONE_INVALID_ROW_CANCELS_IMPORT';

    /**
     * @var string
     */
    private $importMode = self::MODE_ONE_INVALID_ROW_CANCELS_IMPORT;

    /**
     * @var IDataSource
     */
    private $dataSource;

    /**
     * @var IDataInput
     */
    private $dataInput;

    /**
     * @var array<string,array<IValidator>>
     */
    private $validators = [];

    /**
     * @var int
     */
    private $rowNum = 0;

    /**
     * @var bool
     */
    private $allRowsValid = true;

    /**
     * @var array<array>
     */
    private $errors = [];

    /**
     * @param $row
     * @return bool
     */
    protected function checkData(/* Traversable */ $row): bool
    {
        $errorInfo = [];
        foreach ($row as $columnIdentifier => $value) {
            if (empty($this->validators[$columnIdentifier])) {
                continue;
            }

            $errorMessages = [];
            foreach ($this->validators[$columnIdentifier] as $validator) {
                if (!$validator->isValid($value)) {
                    $errorMessages[] = $validator->getMessages();
                }
            }

            if (!empty($errorMessages)) {
                $errorInfo[] = [
                    'col' => $columnIdentifier,
                    'errors' => $errorMessages
                ];
            }
        }

        if (empty($errorInfo)) {
            return true;
        }

        $this->errors[] = [
            'row' => $this->rowNum,
            'errors' => $errorInfo
        ];

        return false;
    }

    /**
     * @param string $importMode
     * @return Import
     */
    public function setImportMode(string $importMode): Import
    {
        $this->importMode = $importMode;

        return $this;
    }

    /**
     * @return string
     */
    public function getImportMode(): string
    {
        return $this->importMode;
    }

    /**
     * @param IDataSource $dataSource
     *
     * @return Import
     */
    public function setDataSource(IDataSource $dataSource): Import
    {
        $this->dataSource = $dataSource;

        return $this;
    }

    /**
     * @param IDataInput $dataInput
     * @return Import
     */
    public function setDataInput(IDataInput $dataInput): Import
    {
        $this->dataInput = $dataInput;

        return $this;
    }

    /**
     * @param IValidator $validator
     * @param string $forColumnIdentifier
     * @return $this
     */
    public function registerValidator(IValidator $validator, string $forColumnIdentifier): Import
    {
        if (!isset($this->validators[$forColumnIdentifier])) {
            $this->validators[$forColumnIdentifier] = [];
        }

        $this->validators[$forColumnIdentifier][] = $validator;

        return $this;
    }

    /**
     * @return array<array>
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * @return Import
     * @throws DataImportException
     */
    public function import(): Import
    {
        $this->allRowsValid = true;
        $this->rowNum = 0;
        $this->errors = [];

        if (!isset($this->dataSource)) {
            throw new DataImportException("Datasource missing.");
        }

        if (!isset($this->dataInput)) {
            throw new DataImportException("Data input missing.");
        }

        $this->dataInput->begin();

        foreach ($this->dataSource->getData() as $rowNum => $row) {
            $this->rowNum++;

            $rowIsValid = $this->checkData($row);
            if (!$rowIsValid) {
                // TODO: Report invalid row;

                $this->allRowsValid = false;
                continue;
            }

            if (!$this->allRowsValid && $this->getImportMode() === static::MODE_ONE_INVALID_ROW_CANCELS_IMPORT) {
                // if we found an invalid row and the import mode is MODE_ONE_INVALID_ROW_CANCELS_IMPORT
                // we omit all subsequent imports.
                continue;
            }

            try {
                $this->dataInput->import($row);
            } catch (\Exception $e) {
                // TODO: Error handling: row was valid but import failed.
                //  -> Report invalid row;
            }
        }

        if ($this->getImportMode() === static::MODE_ONE_INVALID_ROW_CANCELS_IMPORT && !$this->allRowsValid) {
            $this->dataInput->cancel();
            // TODO: Warning
            return $this;
        }

        $this->dataInput->flush();

        return $this;
    }
}
