<?php
/**
 * @copyright softwarewerk <info@softwarewerk.de>
 * @author    chris <chris@softwarewerk.de>
 * Creationtime: 22:54 - 03.04.21
 */

namespace cu\DataImport\DataSource;

use cu\DataImport\DataImportException;

/**
 * Class CSVFileDataSource
 *
 * @package cu\DataImport\DataSource
 */
class CSVFileDataSource implements IFileDataSource, \Iterator
{
    use ColumnSpecTrait;

    public const CSV_DELIMITER_SEMICOLON = ";";
    public const CSV_DELIMITER_COLON = ";";
    public const CSV_ENCLOSURE_SINGLE_QUOTE = "'";
    public const CSV_ENCLOSURE_DOUBLE_QUOTE = "\"";

    static $defaultDelimiter = self::CSV_DELIMITER_COLON;
    static $defaultEnclosure = null;

    /**
     * @var string
     */
    private $sourceFile;

    /**
     * @var ?string
     */
    private $delimiter;

    /**
     * @var ?string
     */
    private $enclosure;

    /**
     * @var array
     */
    private $colSpec;

    /**
     * @var resource
     */
    private $resourceHandle;

    /**
     * @var int
     */
    private $lineNumber = 1;

    /**
     * @var array<string>
     */
    private $currentValue;

    /**
     * CSVFileDataSource constructor.
     *
     * @param ?string $sourceFile
     * @param ?array $colSpec
     * @param ?string $delimiter
     * @param ?string $enclosure
     * @throws DataImportException
     */
    public function __construct(
        ?string $sourceFile,
        ?array $colSpec = null,
        ?string $delimiter = null,
        ?string $enclosure = null
    ) {
        if ($sourceFile) {
            $this->setFile($sourceFile);
        }

        $this->delimiter = $delimiter ?? static::$defaultDelimiter;
        $this->enclosure = $enclosure ?? static::$defaultEnclosure;

        $this->colSpec = $this->assembleColSpec($colSpec);
    }

    /**
     * @param string $sourceFile
     *
     * @return void
     */
    public function setFile(string $sourceFile): void
    {
        $this->sourceFile = $sourceFile;
    }

    /**
     * @inheritDoc
     */
    public function getIterable(): \Traversable
    {
        return $this;
    }

    /**
     * @param string $fileType
     * @return bool
     */
    public function canHandle(string $fileType): bool
    {
        return 'csv' === $fileType;
    }

    /**
     * @inheritDoc
     */
    public function current(): array
    {
        return $this->currentValue;
    }

    /**
     * @inheritDoc
     */
    public function next()
    {
        $this->lineNumber++;
    }

    /**
     * @inheritDoc
     */
    public function key()
    {
        return $this->lineNumber;
    }

    /**
     * @inheritDoc
     */
    public function valid(): bool
    {
        $this->currentValue = [];

        if (!is_resource($this->resourceHandle)) {
            return false;
        }

        // If we got no col spec then iterating the data source makes no sense.
        if (empty($this->colSpec)) {
            return false;
        }

        if (!feof($this->resourceHandle)) {
            $line = fgetcsv($this->resourceHandle, 0, ';');
            if (null !== $line && false !== $line) {
                $values = array_values($line);
                foreach ($this->colSpec as $colName => $colSpec) {
                    $colIndex = @$colSpec['colIndex'];
                    if (!isset($values[$colIndex])) {
                        $this->currentValue[$colName] = null;
                    } else {
                        $this->currentValue[$colName] = $values[$colIndex];
                    }
                }

                return true;
            }
        }

        @fclose($this->resourceHandle);

        unset($this->resourceHandle);

        return false;
    }

    /**
     * @inheritDoc
     */
    public function rewind()
    {
        if (empty($this->sourceFile) || !is_readable($this->sourceFile)) {
            throw new DataImportException('Source file not readable.');
        }

        if (!empty($this->resourceHandle)) {
            @fclose($this->resourceHandle);
            unset($this->resourceHandle);
        }

        $this->resourceHandle = fopen($this->sourceFile, 'r');
        $this->lineNumber = 1;
    }
}
