<?php
/**
 * @copyright softwarewerk <info@softwarewerk.de>
 * @author    chris <chris@softwarewerk.de>
 * Creationtime: 22:54 - 03.04.21
 */

namespace cu\DataImport;

use cu\DataImport\DataInput\IDataInput;
use cu\DataImport\DataSource\IDataSource;
use cu\DataImport\Validator\AbstractValidator;
use cu\DataImport\Validator\IValidator;

/**
 * Class Import
 *
 * @package CuCommon
 */
class Import
{
    public const MODE_IMPORT_VALID_ROWS = 'IMPORT_VALID_ROWS';
    public const MODE_ONE_INVALID_ROW_CANCELS_IMPORT = 'ONE_INVALID_ROW_CANCELS_IMPORT';

    /**
     * @var string
     */
    private $importMode = self::MODE_ONE_INVALID_ROW_CANCELS_IMPORT;

    /**
     * @var IDataSource
     */
    private $dataSource;

    /**
     * @var IDataInput
     */
    private $dataInput;

    /**
     * @var ITranslator
     */
    private $translator;

    /**
     * @var array<string,array<IValidator>>
     */
    private $validators = [];

    /**
     * @var int
     */
    private $rowIndex = 0;

    /**
     * @var bool
     */
    private $allRowsValid = true;

    /**
     * @var array<array>
     */
    private $errors = [];

    /**
     * @return ITranslator
     */
    public function initTranslator(): ITranslator
    {
        return $this->translator = $this->translator instanceof ITranslator
            ? $this->translator
            : new Translator();
    }

    /**
     * @param ITranslator $translator
     * @return Import
     */
    public function setTranslator(ITranslator $translator): Import
    {
        $this->translator = $translator;

        return $this;
    }

    /**
     * @param $row
     * @return bool
     */
    protected function checkData(/* Traversable */ $row): bool
    {
        $errorInfo = [];

        if (empty($row)) {
            $this->errors[] = [
                'rowIndex' => $this->rowIndex,
                'error' => 'No data exported from this row.'
            ];

            return false;
        }

        foreach ($row as $columnIdentifier => $colInfo) {
            if (empty($this->validators[$columnIdentifier])) {
                continue;
            }

            $errorMessages = [];
            foreach ($this->validators[$columnIdentifier] as $validator) {
                if (!$validator->isValid($colInfo['value'])) {
                    $errorMessages[] = $this->initTranslator()->translateKey(
                        $validator->getMessageKey(),
                        [$colInfo['value'], $this->rowIndex, $colInfo['colIndex']]
                    );
                }
            }

            if (!empty($errorMessages)) {
                $errorInfo[$columnIdentifier] = [
                    'colIndex' => $colInfo['colIndex'],
                    'errors' => $errorMessages
                ];
            }
        }

        if (empty($errorInfo)) {
            return true;
        }

        $this->errors[] = [
            'rowIndex' => $this->rowIndex,
            'errors' => $errorInfo
        ];

        return false;
    }

    /**
     * @param string $importMode
     * @return Import
     */
    public function setImportMode(string $importMode): Import
    {
        $this->importMode = $importMode;

        return $this;
    }

    /**
     * @return string
     */
    public function getImportMode(): string
    {
        return $this->importMode;
    }

    /**
     * @param IDataSource $dataSource
     *
     * @return Import
     */
    public function setDataSource(IDataSource $dataSource): Import
    {
        $this->dataSource = $dataSource;

        return $this;
    }

    /**
     * @param IDataInput $dataInput
     * @return Import
     */
    public function setDataInput(IDataInput $dataInput): Import
    {
        $this->dataInput = $dataInput;

        return $this;
    }

    /**
     * @param string $forColumnIdentifier
     * @param IValidator $validator
     * @return $this
     */
    public function registerValidator(string $forColumnIdentifier, IValidator $validator): Import
    {
        if (!isset($this->validators[$forColumnIdentifier])) {
            $this->validators[$forColumnIdentifier] = [];
        }

        $this->validators[$forColumnIdentifier][] = $validator;

        return $this;
    }

    /**
     * @return array<array>
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * @return Import
     * @throws DataImportException
     */
    public function import(): Import
    {
        $this->allRowsValid = true;
        $this->rowIndex = 0;
        $this->errors = [];

        if (!isset($this->dataSource)) {
            throw new DataImportException("Datasource missing.");
        }

        if (!isset($this->dataInput)) {
            throw new DataImportException("Data input missing.");
        }

        $this->dataInput->begin();

        foreach ($this->dataSource->getIterable() as $rowIndex => $row) {
            $this->rowIndex++;

            $rowIsValid = $this->checkData($row);
            if (!$rowIsValid) {
                // TODO: Report invalid row;

                $this->allRowsValid = false;
                continue;
            }

            if (!$this->allRowsValid && $this->getImportMode() === static::MODE_ONE_INVALID_ROW_CANCELS_IMPORT) {
                // if we found an invalid row and the import mode is MODE_ONE_INVALID_ROW_CANCELS_IMPORT
                // we omit all subsequent imports.
                continue;
            }

            try {
                $this->dataInput->import($row);
            } catch (\Exception $e) {
                // TODO: Error handling: row was valid but import failed.
                //  -> Report invalid row;
            }
        }

        if (!$this->allRowsValid && $this->getImportMode() === static::MODE_ONE_INVALID_ROW_CANCELS_IMPORT) {
            $this->dataInput->cancel();
            // TODO: Warning
            return $this;
        }

        $this->dataInput->flush();

        return $this;
    }
}
